/*:
 * @target MZ
 * @plugindesc v1.5【HS】Spine再生コーディネータ：直前スナップ＋生成待ち＋安全ガード（転送は既定OFF／ロード時は凍結解除／待機系は強制ループ）
 * @author HS
 * @help
 * ■ 概要
 *  - Options/Save/Load戻りやMap転送直後の「Spineが止まる／表情が変わる」を、
 *    直前スナップ最優先の再指示＋Spine生成完了待ち＋複数回リトライで解消します。
 *  - 転送（reserveTransfer）由来の復帰は既定で無効（新マップ演出を尊重）。
 *  - ロード直後は timeScale≤0 の“凍結トラック”を検出して再生キック。
 *  - idle 等の待機系はループ強制（正規表現で調整可）。
 *
 * ■ 推奨の並び順（上→下）
 *  1) PKD_Spine2DPlayer.js
 *  2) HS_SpineSnapshotRestore.js（併用推奨・ロード補正の基盤）
 *  3) HS_SpineAfterLoadDedupe.js（任意）
 *  4) ★このプラグイン（いちばん下）
 *
 * @param waitOnReturn
 * @text 戻り後の初期待機フレーム
 * @type number @min 0 @max 60
 * @default 0
 *
 * @param retryCount
 * @text 追加リトライ回数
 * @type number @min 0 @max 120
 * @default 24
 *
 * @param retryInterval
 * @text リトライ間隔（フレーム）
 * @type number @min 1 @max 10
 * @default 1
 *
 * @param resumeOnTransfer
 * @text Map転送でも復帰を試みる
 * @type boolean
 * @default false
 * @desc false: 転送では復帰しない（新マップ演出を尊重）。true: 直前スナップで復帰を試みる。
 *
 * @param waitOnLoadExtra
 * @text ロード直後の追加待機フレーム
 * @type number @min 0 @max 60
 * @default 2
 * @desc ロード→Map到着直後は初期化が重なるため、数フレーム余裕を持ってから復帰判定します。
 *
 * @param fallbackName
 * @text 最終手段のアニメ名（任意）
 * @type string
 * @default
 *
 * @param fallbackLoop
 * @text Fallbackをループ
 * @type boolean
 * @default true
 *
 * @param forceLoopRegex
 * @text ループ強制するアニメ名の正規表現
 * @type string
 * @default ^(idle|wait|breath|loop|base|stand)$
 * @desc この正規表現にマッチするアニメは、スナップがfalseでもloop=trueで再生します（大文字小文字は無視）。
 *
 * @param debugLog
 * @text デバッグログ
 * @type boolean
 * @default true
 */
(() => {
  "use strict";
  const PN = "HS_SpineResumeCoordinator";
  const P  = PluginManager.parameters(PN);

  const WAIT0     = Math.max(0, Number(P.waitOnReturn   || 0));
  const RETRY     = Math.max(0, Number(P.retryCount     || 24));
  const STEP      = Math.max(1, Number(P.retryInterval  || 1));
  const ALLOW_T   = String(P.resumeOnTransfer || "false") === "true";
  const WAIT_LOAD = Math.max(0, Number(P.waitOnLoadExtra || 2));
  const FB_N      = String(P.fallbackName || "");
  const FB_L      = String(P.fallbackLoop || "true") === "true";
  const RE_FORCE  = new RegExp(String(P.forceLoopRegex || "^(idle|wait|breath|loop|base|stand)$"), "i");
  const DEBUG     = String(P.debugLog || "true") === "true";

  const log  = (...a)=>{ if(DEBUG) console.log(`[${PN}]`, ...a); };
  const warn = (...a)=>{ if(DEBUG) console.warn(`[${PN}]`, ...a); };

  // 参照
  const sys  = ()=> $gameSystem;
  const tmp  = ()=> $gameTemp;
  const dict = ()=> (tmp() && tmp()._pSpineAnimations) || null;

  // 専用スナップ倉庫（このプラグインだけの鍵）
  function S(){ sys()._hsSC ||= {}; return sys()._hsSC; }
  function clearStore(){ S().snap = {}; S().reason = ""; S().srcMapId = 0; }

  // トラックから [name, loop, mix, idx] 推定
  function pickFromTracks(spr){
    try{
      const st = spr.getAnimationState && spr.getAnimationState();
      if (!st || !Array.isArray(st.tracks)) return null;
      const ent = st.tracks.find(t => t && t.animation);
      if (!ent || !ent.animation) return null;
      const name = ent.animation.name;
      const loop = !!ent.loop;
      const mix  = Number(ent.mixDuration ?? 0) || 0;
      const idx  = Number(ent.trackIndex  ?? 0) || 0;
      return [name, loop, mix, idx];
    }catch(_){ return null; }
  }
  function hasAnyTrack(spr){
    try{
      const st = spr.getAnimationState && spr.getAnimationState();
      return !!(st && Array.isArray(st.tracks) && st.tracks.some(t => t && t.animation));
    }catch(_){ return false; }
  }
  function isPlaying(spr){
    try{
      const st = spr.getAnimationState && spr.getAnimationState();
      return !!(st && hasAnyTrack(spr) && st.timeScale > 0);
    }catch(_){ return false; }
  }

  // スナップ撮影（Map上から push/transfer 直前）
  function snapshotNow(reasonTag){
    const d = dict();
    if (!d){ log("snapshot: no dict", reasonTag); return; }
    const ids = Object.keys(d).filter(k => !!d[k]);
    if (!ids.length){ log("snapshot: no spines", reasonTag); return; }

    const store = {};
    ids.forEach(id=>{
      const spr = d[id];
      try{
        const st  = spr.getAnimationState && spr.getAnimationState();
        const ent = pickFromTracks(spr);
        store[id] = {
          last: Array.isArray(spr.lastAnimationState) ? spr.lastAnimationState :
                 ent ? ent : null,
          ts: (st && st.timeScale>0) ? st.timeScale : 1
        };
      }catch(_){}
    });

    S().snap     = store;
    S().reason   = reasonTag;
    S().srcMapId = $gameMap ? $gameMap.mapId() : 0;
    log("snapshot ids:", Object.keys(store), `reason=${reasonTag}`, `map=${S().srcMapId}`);
  }

  // 復帰ジョブ
  function ensureJob(scene){
    scene._hsSC_job ||= {
      wait: WAIT0,
      tries: RETRY,
      step: STEP,
      stepCnt: 0,
      done: false,
      reason: S().reason,       // "push:Scene_*" or "transfer" or ""
      srcMapId: S().srcMapId,
      loadFlag: false,          // ロード直後を検知
      doneIds: Object.create(null)
    };
    return scene._hsSC_job;
  }
  function hasAnySpine(){ try{ const d=dict(); return !!(d && Object.keys(d).some(k=>!!d[k])); }catch(_){ return false; } }

  // ループ強制を含む setAnimation
  function setAnimForce(spr, arr, ts){
    try{
      const st = spr.getAnimationState && spr.getAnimationState();

      const name = arr && arr[0] || "";
      const hintedLoop = !!(arr && arr[1]);
      const wantLoop   = hintedLoop || RE_FORCE.test(name); // 待機系は強制ループ

      if (st) st.timeScale = Math.max(1, Number(ts||1));

      if (spr && spr.setAnimation && name){
        const mix = Number(arr && arr[2] || 0) || 0;
        const idx = Number(arr && arr[3] || 0) || 0;
        spr.setAnimation(name, wantLoop, mix, idx);
        // 念押し：trackEntry.loop を直接trueに（pixi-spineはここを見ることがある）
        try{
          const entry = st && st.tracks ? (st.tracks[idx] || st.tracks[0]) : null;
          if (entry) entry.loop = wantLoop;
        }catch(_){}
      }

      const sp = spr.getSpineObject && spr.getSpineObject();
      if (sp){ sp.autoUpdate = true; if (sp.update) sp.update(0); }

      return true;
    }catch(e){
      warn("setAnimForce err", e);
      return false;
    }
  }

  function tryResumeOnce(scene){
    const d = dict(); if (!d) return false;
    const ids = Object.keys(d).filter(k => !!d[k]); if (!ids.length) return false;

    const snap = S().snap || {};
    let any = false;

    for (const id of ids){
      if (scene._hsSC_job.doneIds[id]) continue;

      const spr = d[id];

      // 再生中（tracksあり & timeScale>0）は尊重して何もしない
      if (isPlaying(spr)){
        scene._hsSC_job.doneIds[id] = true;
        log("skip (already playing):", id);
        continue;
      }
      // 転送理由で、かつロードでない場合は“何もしない”（新マップ演出尊重）
      if (scene._hsSC_job.reason === "transfer" && !scene._hsSC_job.loadFlag && !ALLOW_T){
        scene._hsSC_job.doneIds[id] = true;
        log("skip (transfer w/o load):", id);
        continue;
      }
      // ここに来た場合：
      //  - tracksが無い もしくは
      //  - tracksは有るが timeScale<=0（凍結）→ 復帰対象

      // 再生指示：snap → sprite.last → tracks → fallback
      let arr = null, ts = 1;
      if (snap[id] && Array.isArray(snap[id].last)){ arr = snap[id].last; ts = snap[id].ts || 1; log("use SNAP:", id, arr); }
      if (!arr && Array.isArray(spr.lastAnimationState)){ arr = spr.lastAnimationState; log("use SPRITE.last:", id, arr); }
      if (!arr){ const tr = pickFromTracks(spr); if (tr){ arr = tr; log("use TRACKS:", id, arr); } }
      if (!arr && FB_N){ arr = [FB_N, FB_L, 0, 0]; log("use FALLBACK:", id, arr); }

      if (arr){
        const ok = setAnimForce(spr, arr, ts);
        scene._hsSC_job.doneIds[id] = true;
        any = ok || any;
      }
      // arrが未決なら次の試行へ
    }
    return any;
  }

  // push直前にスナップ（Map上のみ）
  const _push = SceneManager.push;
  SceneManager.push = function(sceneClass){
    try{
      const cur = this._scene;
      const target = sceneClass && sceneClass.name;
      if (cur instanceof Scene_Map){
        if (target==="Scene_Options" || target==="Scene_Save" || target==="Scene_Load" ||
            target==="Scene_File"   || target==="Scene_GameEnd"){
          snapshotNow(`push:${target}`);
        }
      }
    }catch(e){ warn(e); }
    return _push.call(this, sceneClass);
  };

  // 転送予約時にもスナップ（既定では後で使わない）
  const _reserveTransfer = Game_Player.prototype.reserveTransfer;
  Game_Player.prototype.reserveTransfer = function(mapId, x, y, d, fadeType){
    try{ snapshotNow("transfer"); }catch(e){ warn(e); }
    return _reserveTransfer.call(this, mapId, x, y, d, fadeType);
  };

  // Map側で復帰ジョブ
  const _start = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function(){
    _start.call(this);
    const job = ensureJob(this);

    // ロード直後を検知（HS_SpineSnapshotRestore が立てるフラグに対応）
    try{
      if (sys()?._hsLastLoadOK || sys()?._hsPendResumeAfterLoad){
        job.loadFlag = true;
        job.wait += WAIT_LOAD;   // ロード時は余裕を持って判定
        log("load context detected → extra wait:", WAIT_LOAD, "total wait:", job.wait);
      }
    }catch(_){}

    // 転送スナップは、ロードでない限り既定で使わない
    if (job.reason === "transfer" && !job.loadFlag && !ALLOW_T){
      job.done = true;
      log("skip resume (reason=transfer, resumeOnTransfer=false)");
      clearStore();
      return;
    }
    log("resume job start:", JSON.stringify({wait:job.wait,tries:job.tries,step:job.step,reason:job.reason,src:job.srcMapId,load:job.loadFlag}));
  };

  const _update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function(){
    _update.call(this);
    const job = ensureJob(this);
    if (job.done) return;

    if (job.wait > 0){ job.wait--; return; }
    if (!hasAnySpine()) return;                 // Spine未生成なら待つ
    if (job.stepCnt++ % job.step !== 0) return; // 間引き

    const ok = tryResumeOnce(this);

    // すべてのIDが確定（再生中 or doneIds）したら終了
    const d = dict() || {};
    const allDone = Object.keys(d).every(id => job.doneIds[id] || isPlaying(d[id]));
    if (ok || allDone){
      job.done = true;
      clearStore();
      // ロード系フラグを穏やかに解放（SnapshotRestore側と二重でも害なし）
      if (sys()){ sys()._hsLastLoadOK = false; sys()._hsPendResumeAfterLoad = false; }
      log("resume done:", { ok, allDone });
    }else{
      if (job.tries-- <= 0){
        job.done = true;
        clearStore();
        warn("resume exhausted (no-op)");
      }
    }
  };
})();

